#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
自动重启服务 - 测试脚本
此脚本用于测试自动重启服务的基本功能
"""

import os
import json
import time
import logging
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
import smtplib

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('reboot_service_test.log'),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger('reboot_service_test')

def load_config():
    """加载配置文件"""
    config_path = 'reboot_config.json'
    
    # 如果配置文件不存在，从示例文件复制
    if not os.path.exists(config_path):
        if os.path.exists('reboot_config_example.json'):
            logger.info('配置文件不存在，从示例文件复制')
            with open('reboot_config_example.json', 'r', encoding='utf-8') as f:
                # 读取文件内容并移除注释
                content = f.read()
                # 移除 /* */ 格式的注释
                import re
                content = re.sub(r'\/\*.*?\*\/', '', content, flags=re.DOTALL)
                # 移除 // 格式的注释（如果有）
                content = re.sub(r'//.*$', '', content, flags=re.MULTILINE)
                # 移除空行
                content = '\n'.join(line.strip() for line in content.splitlines() if line.strip())
                config = json.loads(content)
            
            # 使用默认测试配置
            config['email']['smtp_server'] = 'smtp.example.com'
            config['email']['smtp_port'] = 587
            config['email']['username'] = 'test@example.com'
            config['email']['password'] = 'test_password'
            config['email']['sender'] = 'test@example.com'
            config['email']['recipients'] = ['admin@example.com']
            
            # 保存测试配置
            with open(config_path, 'w', encoding='utf-8') as f:
                json.dump(config, f, indent=4, ensure_ascii=False)
            
            logger.info(f'已创建测试配置文件: {config_path}')
            return config
        else:
            logger.error('配置文件和示例文件都不存在')
            return None
    
    # 加载现有配置文件
    try:
        with open(config_path, 'r', encoding='utf-8') as f:
            config = json.load(f)
        logger.info('配置文件加载成功')
        return config
    except Exception as e:
        logger.error(f'加载配置文件失败: {e}')
        return None

def send_test_email(config):
    """发送测试邮件"""
    if not config or 'email' not in config:
        logger.error('配置文件无效')
        return False
    
    email_config = config['email']
    
    try:
        logger.info(f'准备发送测试邮件，SMTP服务器: {email_config["smtp_server"]}')
        
        # 创建邮件内容
        msg = MIMEMultipart()
        msg['From'] = email_config['sender']
        msg['To'] = ', '.join(email_config['recipients'])
        msg['Subject'] = '【测试】自动重启服务邮件通知功能'
        
        body = """
        这是自动重启服务的测试邮件。
        
        如果您收到此邮件，说明邮件发送功能正常工作。
        
        自动重启服务将在15天后重启系统，并在重启前发送邮件通知。
        
        时间戳: %s
        """ % time.strftime('%Y-%m-%d %H:%M:%S')
        
        msg.attach(MIMEText(body, 'plain', 'utf-8'))
        
        # 模拟发送邮件（实际环境中取消注释下面的代码）
        logger.info('模拟邮件发送成功！收件人: %s', email_config['recipients'])
        logger.info('注意：这是测试模式，实际邮件并未发送。在生产环境中，请取消代码中的注释。')
        
        # 实际发送邮件的代码（生产环境中启用）
        # with smtplib.SMTP(email_config['smtp_server'], email_config['smtp_port']) as server:
        #     server.starttls()
        #     server.login(email_config['username'], email_config['password'])
        #     server.send_message(msg)
        # logger.info('邮件发送成功')
        
        return True
        
    except Exception as e:
        logger.error(f'发送邮件失败: {e}')
        return False

def test_reboot_logic():
    """测试重启逻辑"""
    logger.info('开始测试重启逻辑')
    
    # 模拟重启时间计算
    status_file = 'reboot_status.json'
    current_time = time.time()
    
    # 如果状态文件不存在，创建新的
    if not os.path.exists(status_file):
        logger.info('状态文件不存在，创建新的启动记录')
        status = {
            'last_boot_time': current_time,
            'next_reboot_time': current_time + (15 * 24 * 3600)  # 15天后
        }
        with open(status_file, 'w', encoding='utf-8') as f:
            json.dump(status, f, indent=4)
        logger.info('状态文件创建成功')
    else:
        # 读取现有状态文件
        try:
            with open(status_file, 'r', encoding='utf-8') as f:
                status = json.load(f)
            logger.info('状态文件加载成功')
            logger.info(f'上次启动时间: {time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(status["last_boot_time"]))}')
            logger.info(f'下次重启时间: {time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(status["next_reboot_time"]))}')
        except Exception as e:
            logger.error(f'加载状态文件失败: {e}')
            return False
    
    # 计算剩余时间
    time_left = status['next_reboot_time'] - current_time
    days_left = time_left / (24 * 3600)
    
    logger.info(f'距离下次重启还有: {days_left:.2f} 天')
    logger.info(f'相当于: {time_left//3600} 小时 {time_left%3600//60} 分钟 {time_left%60:.0f} 秒')
    
    # 模拟警告时间检查
    warning_hours = 24  # 提前24小时警告
    warning_minutes = 30  # 提前30分钟最终警告
    
    if time_left <= warning_hours * 3600:
        logger.info(f'⚠️  即将在 {days_left:.2f} 天后重启，需要发送警告邮件')
    if time_left <= warning_minutes * 60:
        logger.info(f'⚠️  即将在 {time_left//60} 分钟后重启，需要发送最终警告邮件')
    if time_left <= 0:
        logger.info('⚠️  到达重启时间，执行重启操作')
    
    logger.info('重启逻辑测试完成')
    return True

def main():
    """主函数"""
    logger.info('====================================')
    logger.info('  自动重启服务测试脚本')
    logger.info('====================================')
    
    # 加载配置
    config = load_config()
    if not config:
        logger.error('无法加载配置，测试中止')
        return 1
    
    # 测试邮件功能
    logger.info('\n=== 测试邮件功能 ===')
    email_result = send_test_email(config)
    if email_result:
        logger.info('[OK] 邮件功能测试通过')
    else:
        logger.error('[ERROR] 邮件功能测试失败')
    
    # 测试重启逻辑
    logger.info('\n=== 测试重启逻辑 ===')
    reboot_result = test_reboot_logic()
    if reboot_result:
        logger.info('[OK] 重启逻辑测试通过')
    else:
        logger.error('[ERROR] 重启逻辑测试失败')
    
    logger.info('\n=== 测试总结 ===')
    if email_result and reboot_result:
        logger.info('[OK] 所有测试通过！自动重启服务基本功能正常')
        logger.info('您可以通过运行 install_service.py 安装服务')
        return 0
    else:
        logger.error('[ERROR] 部分测试失败，请检查日志并修复问题')
        return 1

if __name__ == '__main__':
    exit_code = main()
    print('\n测试完成，请查看日志获取详细信息')
    print('按Enter键退出...')
    input()
    exit(exit_code)